/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.contrib.application.task.test.po;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.openqa.selenium.By;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.interactions.Actions;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.xwiki.test.ui.po.ViewPage;

/**
 * Can perform common test actions on a div generated by a taskgantt macro.
 */
public class TaskManagerGanttMacro extends ViewPage
{
    public static final List<String> viewModes =
        Arrays.asList("Quarter Day", "Half Day", "Day", "Week", "Month", "Year");

    private final WebElement ganttElement;

    public TaskManagerGanttMacro(WebElement ganttElement)
    {
        this.ganttElement = ganttElement;
        // Wait for tasks to load.
        getDriver().waitUntilCondition(
            ExpectedConditions.presenceOfElementLocated(By.cssSelector(".gantt-container, .box")));
    }

    static public List<TaskManagerGanttMacro> getGanttMacrosOnCurrentPage()
    {
        return getUtil().getDriver().findElements(By.cssSelector("div[id^='taskgantt-gantt']")).stream()
            .map((WebElement ganttElement) -> {
                return new TaskManagerGanttMacro(ganttElement);
            }).collect(Collectors.toList());
    }

    /**
     * Changes the time scale of the gantt diagram ('Month', 'Day', 'Quarter Day', etc.).
     *
     * @param viewMode the value of the desired view mode.
     */
    public void changeViewMode(String viewMode)
    {
        ganttElement.findElement(By.cssSelector(".view-mode-button[value='" + viewMode + "']")).click();
    }

    /**
     * Whether the overlay preventing any interaction with the diagram is active.
     *
     * @return {@code true} if the overlay is active.
     */
    public boolean isOverlayActive()
    {
        return (!ganttElement.findElements(By.cssSelector(".taskgantt-noedit-overlay.active")).isEmpty());
    }

    /**
     * Get the gantt bar element of the given task.
     *
     * @param id the id of the bar task element, given as a full page reference ("xwiki:Space.TaskPage")
     * @return the gantt bar element if it exists.
     */
    public WebElement getTaskBar(String id) throws NoSuchElementException
    {
        return this.getSvgGantt().findElement(By.cssSelector(".bar-wrapper[data-id='" + id + "']"));
    }

    /**
     * Get all task ids of tasks present in the gantt diagram.
     *
     * @return a list of ids of all tasks in the gantt diagram. The task ids are page references in the
     *     'xwiki:Space.Page' format.
     */
    public List<String> getTaskIds()
    {
        if (this.getSvgGantt() != null) {
            return this.getSvgGantt().findElements(By.cssSelector(".bar-wrapper")).stream()
                .map((elem) -> elem.getAttribute("data-id")).collect(Collectors.toList());
        } else {
            return List.of();
        }
    }

    /**
     * Double click a task, which opens the task page in a new tab.
     *
     * @param id the id of the bar task element, given as a full page reference ("xwiki:Space.TaskPage")
     */
    public void openTaskPage(String id)
    {
        WebElement taskBar = this.getTaskBar(id).findElement(By.cssSelector(".bar"));
        Actions builder = new Actions(getDriver().getWrappedDriver());
        builder.moveToElement(this.ganttElement).doubleClick(taskBar).perform();
    }

    /**
     * Drag the task start date by the specified amount of pixels.
     *
     * @param id the task id to drag
     * @param offset amount of pixels to move. Negative values move to the left, positive to the right. Note that
     *     the date will not be changed if the amount to move is less than the snapping interval defined in the gantt's
     *     js (also depends on the current view mode).
     */
    public void dragTaskStart(String id, int offset)
    {
        WebElement taskBar = this.getTaskBar(id);
        WebElement startHandle = taskBar.findElement(By.cssSelector(".handle.left"));
        dragElementHorizontal(startHandle, offset);
    }

    /**
     * Drag the task end date by the specified amount of pixels.
     *
     * @param id the task id to drag
     * @param offset amount of pixels to move. Negative values move to the left, positive to the right. Note that
     *     the date will not be changed if the amount to move is less than the snapping interval defined in the gantt's
     *     js (also depends on the current view mode).
     */
    public void dragTaskEnd(String id, int offset)
    {
        WebElement taskBar = this.getTaskBar(id);
        WebElement endHandle = taskBar.findElement(By.cssSelector(".handle.right"));
        dragElementHorizontal(endHandle, offset);
    }

    /**
     * Drag the task progress by the specified amount of pixels.
     *
     * @param id the task id to drag
     * @param offset amount of pixels to move. Negative values move to the left, positive to the right.
     * @return false if the bar doesn't have a progress handle (either 0 or 100 progress), true if the progress was
     *     changed.
     */
    public boolean dragProgress(String id, int offset)
    {
        WebElement taskBar = this.getTaskBar(id);
        try {
            WebElement progressHandle = taskBar.findElement(By.cssSelector(".handle.progress"));
            dragElementHorizontal(progressHandle, offset);
            return true;
        } catch (NoSuchElementException e) {
            return false;
        }
    }

    /**
     * Drag the task by the specified amount of pixels.
     *
     * @param id the task id to drag
     * @param offset amount of pixels to move. Negative values move to the left, positive to the right. Note that
     *     the date will not be changed if the amount to move is less than the snapping interval defined in the gantt's
     *     js (also depends on the current view mode).
     */
    public void dragTask(String id, int offset)
    {
        WebElement taskBar = this.getTaskBar(id).findElement(By.cssSelector(".bar"));
        dragElementHorizontal(taskBar, offset);
    }

    private WebElement getSvgGantt()
    {
        List<WebElement> svgGantt = ganttElement.findElements(By.cssSelector("svg.gantt"));
        return svgGantt.isEmpty() ? null : svgGantt.get(0);
    }

    private void dragElementHorizontal(WebElement element, int offset)
    {
        Actions dragAndDrop =
            new Actions(getDriver().getWrappedDriver()).moveToElement(this.ganttElement).clickAndHold(element)
                .moveByOffset(offset, 0).release();
        dragAndDrop.perform();
    }
}
